<?php
/**
 * Submit Field Value
 *
 * @package     AutomatorWP\Integrations\ARForms\Triggers\Submit_Field_Value
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_ARForms_Submit_Field_Value extends AutomatorWP_Integration_Trigger {

    public $integration = 'arforms';
    public $trigger = 'arforms_submit_field_value';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User submits a specific field value on a form', 'automatorwp-pro' ),
            'select_option'     => __( 'User submits <strong>a specific field value</strong> on a form', 'automatorwp-pro' ),
            /* translators: %1$s: Field name. %2$s: Field value. %3$s: Post title. %4$s: Number of times. */
            'edit_label'        => sprintf( __( 'User submits the %1$s field with the value %2$s on %3$s %4$s time(s)', 'automatorwp-pro' ), '{field_name}', '{field_value}', '{form}', '{times}' ),
            /* translators: %1$s: Field name. %2$s: Field value. %3$s: Post title. */
            'log_label'         => sprintf( __( 'User submits the %1$s field with the value %2$s on %3$s', 'automatorwp-pro' ), '{field_name}', '{field_value}', '{form}' ),
            'action'            => array(
                'arfliteentryexecute',
                'arfentryexecute'
            ),
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 4,
            'options'           => array(
                'field_name' => array(
                    'from' => 'field_name',
                    'default' => __( 'field name', 'automatorwp' ),
                    'fields' => array(
                        'field_name' => array(
                            'name' => __( 'Field name:', 'automatorwp' ),
                            'type' => 'text',
                            'default' => ''
                        )
                    )
                ),
                'field_value' => array(
                    'from' => 'field_value',
                    'default' => __( 'field value', 'automatorwp' ),
                    'fields' => array(
                        'field_value' => array(
                            'name' => __( 'Field value:', 'automatorwp' ),
                            'type' => 'text',
                            'default' => ''
                        )
                    )
                ),
                'form' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'form',
                    'name'              => __( 'Form:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any form', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_arforms_get_forms',
                    'options_cb'        => 'automatorwp_arforms_options_cb_form',
                    'default'           => 'any'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                array(
                    'form_field:FIELD_NAME' => array(
                        'label'     => __( 'Form field value', 'automatorwp-pro' ),
                        'type'      => 'text',
                        'preview'   => __( 'Form field value, replace "FIELD_NAME" by the field name', 'automatorwp-pro' ),
                    ),
                ),
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param array     $params 
     * @param int       $form_id    
     * @param int       $form   
     * @param array     $item_meta_values   
     */
    public function listener( $params, $errors, $form, $item_meta_values ) {

        $user_id = get_current_user_id();

        // Bail if user is not logged in
        if( $user_id === 0 ) {
            return;
        }

        $form_fields = automatorwp_arforms_get_form_fields_values( $form->id, $item_meta_values );

        foreach( $form_fields as $name => $value ) {

            // Skip field
            if( ! isset( $name ) && ! isset( $value ) ) {
                continue;
            }

            $field_name = $name;
            $field_value = $value;
			
            // Trigger submit form field value event
            automatorwp_trigger_event( array(
                'trigger'       => $this->trigger,
                'user_id'       => $user_id,
                'form_id'       => $form->id,
                'field_name'    => $field_name,
                'field_value'   => $field_value,
                'form_fields'   => $form_fields,
            ) );
        }

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post, field name and value are not received
        if( ! isset( $event['form_id'] ) && ! isset( $event['field_name'] ) && ! isset( $event['field_value'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( $trigger_options['form'] !== 'any' &&  absint( $trigger_options['form'] ) !== absint( $event['form_id'] ) ) {
            return false;
        }

        // Don't deserve if field name doesn't match with the trigger option
        if( $event['field_name'] !== $trigger_options['field_name'] ) {
            return false;
        }

        // Check if field value matches the required one (with support for arrays)
        if( is_array( $event['field_value'] ) ) {

            if( ! in_array( $trigger_options['field_value'], $event['field_value'] ) ) {
                return false;
            }

        } else {

            if( $event['field_value'] !== $trigger_options['field_value'] ) {
                return false;
            }

        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['form_fields'] = ( isset( $event['form_fields'] ) ? $event['form_fields'] : array() );

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an trigger
        if( $log->type !== 'trigger' ) {
            return $log_fields;
        }

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return $log_fields;
        }

        $log_fields['form_fields'] = array(
            'name' => __( 'Fields Submitted', 'automatorwp-pro' ),
            'desc' => __( 'Information about the fields values sent on this form submission.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_ARForms_Submit_Field_Value();